<?php

namespace App;

use App\Models\Acl\role;
use App\Models\Organization\organization;
use App\Models\Region\city;
use App\Models\Service\serviceCard;
use App\Models\Service\serviceCardStatus;
use App\Models\Skill\skill;
use App\Models\Ticket\ticket;
use App\Models\Wallet\Wallet;
use Illuminate\Notifications\Notifiable;
use Illuminate\Foundation\Auth\User as Authenticatable;

class User extends Authenticatable
{
    use Notifiable;

    protected $fillable = [
        'city_id',
        'userName',
        'email',
        'firstName',
        'lastName',
        'fatherName',
        'gender',
        'nationalCode',
        'certificateCode',
        'postalCode',
        'birthday',
        'birthplace',
        'mobile',
        'telephone',
        'address',
        'education',
        'fieldOfStudy',
        'job',
        'saveSkill',
        'api_token',
        'canEdit',
        'isActive',
        'isPay',
        'isAdmin',
        'type',
        'parent',
        'ref',
        'educationPicture',
        'personalPicture',
        'nationalCardPicture',
        'mobile_verified_at',
        'email_verified_at',
        'password',
    ];

    protected $hidden = [
        'password', 'remember_token',
    ];

    protected $casts = [
        'email_verified_at' => 'datetime',
    ];

    protected $appends = ['genderApi','educationApi'];



    // Relations

    public function role()
    {
        return $this->belongsToMany(role::class,'role_user');
    }

    public function settings()
    {
        return $this->hasMany(settingUser::class);
    }

    public function tickets()
    {
        return $this->hasMany(ticket::class);
    }

    public function ticketReferer()
    {
        return $this->belongsToMany(ticket::class,'ticket_refers');
    }

    public function serviceCard()
    {
        return $this->belongsToMany(serviceCard::class,'serviceCard_user')->withTimestamps();
    }

    public function serviceCardStatus()
    {
        return $this->belongsToMany(serviceCardStatus::class,'serviceCardStatus_user');
    }

    public function skill()
    {
        return $this->belongsToMany(skill::class,'skill_user');
    }

    public function city()
    {
        return $this->belongsTo(city::class);
    }

    public function referalOrganizion()
    {
        return $this->belongsToMany(organization::class, 'referalCode_organization')->withTimestamps();
    }

    public function referalUser()
    {
        return $this->belongsToMany(User::class, 'referalCode_user', 'ref_id')->withTimestamps();
    }

    public function organization()
    {
        return $this->belongsToMany(organization::class, 'organization_admin')->withTimestamps();
    }

    public function wallet()
    {
        return $this->morphOne(Wallet::class, 'model');
    }


    // Acl

    public function owns($related)
    {
        return $this->id == $related->user_id;
    }

    public function hasRole($role)
    {
        if (is_string($role)) {
            return role::contains('name', $role);
        }

        return !! $role->intersect($this->role)->count();
    }

    public function getPermissions()
    {
        return auth()->user()->role()->with('permissions')->get()->pluck('permissions','id')->unique('id')->flatten();
    }

    public function allRoles()
    {
        $haveNotRole = role::whereHas('permissions', function ($query){
            $query->whereNotIn('permissions.id',$this->getPermissions()->pluck('id'));
        })->select('roles.id')->get()->pluck('id');
        return role::whereNotIn('id',$haveNotRole);
    }

    public function getWallet()
    {
        if ($this->wallet) {
            return $this->wallet;
        } else {
            $wallet = Wallet::create([
                'model_type' => 'App\User',
                'model_id' => $this->id,
                'withdraw' => 0,
                'deposit' => 0,
                'amount' => 0,
                'status' => 1
            ]);

            return $wallet;
        }
    }


    // Scoeps

    public function scopeSearch($query)
    {
        $request = request();
        $fillables = array_merge($this->getFillable(), ['id']);
        foreach ($request->all() as $key => $value) {
            if (str_contains($key, 'orderby') && (array_search($keyResult = str_replace('orderby_', '', $key), $fillables) !== false)) {
                $query->orderBy($keyResult, $value);
            }
        }

        $query->where(function ($query) use ($request,$fillables) {
            foreach ($request->all() as $key => $value) {
                if (array_search($key, $fillables) !== false) {
                    if ($key != 'api_token') {
                        $query->where($key, $value);
                    }
                } elseif (str_contains($key,'like') && (array_search($keyResult = str_replace('like_', '', $key), $fillables) !== false)) {
                    $query->where($keyResult, 'like', "%{$value}%");
                }
            }

            if ($request->has('query')) {
                $query->orWhere('id', 'like', "%{$request->input('query')}%");
                $query->orWhere('email', 'like', "%{$request->input('query')}%");
                $query->orWhere('firstName', 'like', "%{$request->input('query')}%");
                $query->orWhere('lastName', 'like', "%{$request->input('query')}%");
                $query->orWhere('mobile', 'like', "%{$request->input('query')}%");
                $query->orWhere('nationalCode', 'like', "%{$request->input('query')}%");
            }


            if ($request->has('date_start')) {
                $query->whereDate('created_at', '>=', "{$request->input('start')}");
            }

            if ($request->has('date_end')) {
                $query->whereDate('created_at', '<=', "{$request->input('end')}");
            }

            if ($request->has('paystatus')) {
                $query->has('serviceCard');
            }


            if ($request->has('active')) {
                if ($request->input('active') == 'agree') {
                    $query->where('isActive',1);
                } else {
                    $query->where('isActive','<>',1);
                }
            }


        });

        return $query;
    }

    public function scopeSearchFront($query)
    {
        $request = request();
        $fillables = array_merge($this->getFillable(), ['id']);

        $query->where(function ($query) use ($request,$fillables) {

            if ($request->input('city_id')) {
                $query->where('city_id', $request->input('city_id'));
            }

            if ($request->input('state_id')) {
                $query->whereNotNull('city_id')->whereHas('city', function ($query) {
                    $query->where('state_id', request()->input('state_id'));
                });
            }

            if ($request->input('skill_id')) {
                $query->whereHas('skill', function ($query) {
                    $query->where('skills.id', request()->input('skill_id'));
                });
            }

        });
        return $query;
    }


    // Attributes

    public function getGenderApiAttribute()
    {
        if ($this->gender == 'male') {
            return 'مرد';
        } else {
            return 'زن';
        }
    }

    public function getEducationApiAttribute()
    {
        if ($this->education == 1) {
            return 'زیر دیپلم';
        } elseif ($this->education == 2) {
            return 'دیپلم';
        } elseif ($this->education == 3) {
            return 'کاردانی';
        } elseif ($this->education == 4) {
            return 'کارشناسی';
        } elseif ($this->education == 5) {
            return 'کارشناسی ارشد';
        } elseif ($this->education == 6) {
            return 'دکتری';
        }
    }


    // Extra Methods

    public function getBirthday()
    {
        if ($this->birthday) {
            $year =(integer) explode('-', $this->birthday)[0];
            return (now()->year -  $year) ;
        }

        return 0;
    }
}
