<?php

namespace App\Http\Controllers\Admin\Acl;

use App\Http\Controllers\Controller;
use App\Models\Acl\permission;
use App\Models\Acl\role;
use Illuminate\Http\Request;
use Illuminate\Support\Facades\Gate;

class RoleController extends Controller
{
    public function index()
    {
        $this->authorize('role_index');

        if (request()->has('page')){
            $list = role::whereIn('id',auth()->user()->allRoles()->get()->pluck('id'))->search()->withCount('permissions')->latest()->paginate();

            return response()->json($list);
        }

        return view('admin.acl.role.index');
    }

    public function create()
    {
        $this->authorize('role_store');

        $permissions = auth()->user()->getPermissions();

        return view('admin.acl.role.create',compact('permissions'));
    }

    public function store(Request $request)
    {
        $this->authorize('role_store');

        $this->validate($request, [
            'permission_id.*' => 'exists:permissions,id',
            'name'           => 'required|string|max:255',
        ]);
        role::create($request->all())->permissions()->attach($request->input('permissions'));
        cache()->forever('permission', permission::with('role')->get());

        message();
        return redirect()->route('admin.roles.index');
    }

    public function show(role $role)
    {
        $this->authorize('role_show');

        return response()->json($role);
    }

    public function edit(role $role)
    {
        $this->authorize('role_update');

        $permissions = auth()->user()->getPermissions();
        $rolePermissions = $role->permissions()->get();

        return view('admin.acl.role.edit',compact('role','permissions','rolePermissions'));
    }

    public function update(Request $request, role $role)
    {
        $this->authorize('role_update');

        $this->validate($request, [
            'permissions.*' => 'exists:permissions,id',
            'name'            => 'required|string|max:255',
        ]);
        $permissions = auth()->user()->getPermissions()->pluck('id')->flatten()->intersect($request->input('permissions'));

        $role->update($request->all());
        $role->permissions()->sync($permissions);
        cache()->forever('permission', permission::with('role')->get());

        message('ویرایش شد');
        return redirect()->route('admin.roles.edit',$role->id);
    }

    public function destroy(role $role)
    {
        $this->authorize('role_destroy');

        return response()->json($role->delete());
    }


    //custom methods
    public function search(Request $request)
    {
        $this->authorize('role_index');
        $this->validate($request, [
            'query'=> 'string|max:255'
        ]);
        $list = auth()->user()->allRoles()->where(function ($query) use ($request){
            if ($request->has('query')) {
                $query->orWhere('name','like',"%{$request->input('query')}%")->get();
                $query->orWhere('label','like',"%{$request->input('query')}%")->get();
            }
        })->select('id','name','label')->get();
        return response()->json($list);
    }

    public function getRolePermissions(role $role)
    {
        $this->authorize('role_update');

        return response()->json($role->permissions()->get());
    }

    public function listRole()
    {
        if (Gate::denies('role_index')  || Gate::denies('acl_list')) {
            abort('403');
        }
        return response()->json(auth()->user()->allRoles()->get());
    }
}
